package com.example.los_chatos.network

import com.google.gson.GsonBuilder
import com.google.gson.annotations.SerializedName
import retrofit2.Response
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.http.* 

// --- Data Models for API --- //

// Generic response for simple success/error messages
data class ApiResponse(
    val status: String,
    val message: String,
    val role: String? // Campo añadido para el rol
)

// Request body for user registration
data class RegisterBody(
    val username: String,
    val fullName: String,
    val email: String,
    val password: String
)

// Request body to update a grade
data class GradeUpdateBody(
    @SerializedName("grade_id") val gradeId: Int,
    @SerializedName("grade_value") val gradeValue: String
)

// Data class for a single student
data class Student(
    val id: Int,
    val username: String,
    @SerializedName("full_name") val fullName: String,
    val email: String
)

data class Grade(
    val id: Int, // <-- ID de la nota añadido
    @SerializedName("grade_name") val name: String,
    @SerializedName("grade_value") val value: String
)

data class CourseWithGrades(
    val name: String,
    val professor: String,
    val description: String,
    val grades: List<Grade>
)

// --- Report Data Models ---
data class ReportGrade(
    @SerializedName("grade_name") val gradeName: String,
    @SerializedName("grade_value") val gradeValue: String
)

data class ReportCourse(
    val name: String,
    val professor: String,
    val grades: List<ReportGrade>
)

data class ReportStudent(
    val fullName: String,
    val username: String,
    val courses: List<ReportCourse>
)
// --------------------------

data class ScheduleItem(
    @SerializedName("course_name") val courseName: String,
    @SerializedName("start_time") val startTime: String,
    @SerializedName("end_time") val endTime: String,
    val classroom: String
)

data class UserProfile(
    val username: String,
    @SerializedName("full_name") val fullName: String,
    val email: String
)

// --- Retrofit API Service Interface --- //

interface ApiService {
    @FormUrlEncoded
    @POST("login.php")
    suspend fun login(
        @Field("username") username: String,
        @Field("password") password: String
    ): Response<ApiResponse>

    @POST("register_user.php")
    suspend fun registerUser(@Body body: RegisterBody): Response<ApiResponse>

    @POST("editar_nota.php")
    suspend fun updateGrade(@Body body: GradeUpdateBody): Response<ApiResponse>

    @GET("get_semesters.php")
    suspend fun getSemesters(): Response<List<String>>

    @GET("get_students.php")
    suspend fun getStudents(): Response<List<Student>>

    @GET("generar_reporte.php")
    suspend fun getReportData(@Query("semester") semester: String): Response<List<ReportStudent>>

    @GET("get_semester_details.php")
    suspend fun getSemesterDetails(
        @Query("username") username: String,
        @Query("semester") semester: String
    ): Response<List<CourseWithGrades>>

    @GET("get_schedule.php")
    suspend fun getSchedule(@Query("username") username: String): Response<Map<String, List<ScheduleItem>>>

    @GET("get_profile.php")
    suspend fun getProfile(@Query("username") username: String): Response<UserProfile>
}

// --- Retrofit Client Singleton --- //

object RetrofitClient {
    // Special IP for Android Emulator to connect to the host machine's localhost
    private const val BASE_URL = "https://frank.grupodvh.com//api_notas/"

    val instance: ApiService by lazy {
        val gson = GsonBuilder()
            .setLenient()
            .create()

        val retrofit = Retrofit.Builder()
            .baseUrl(BASE_URL)
            .addConverterFactory(GsonConverterFactory.create(gson))
            .build()
        retrofit.create(ApiService::class.java)
    }
}
